//
//  Helper.swift
//
//

import Foundation
import MBProgressHUD
import OSLog
import UIKit

class Helper {

    // MARK: - Internet connectivity

    static func checkInternetConnection(giveErrorMessage _: Bool? = true) -> Bool {
        let reachability = Reachability()!

        if !reachability.isReachable {
            showErrorMessage(strMessage: "no_connection_message".localized)
        }
        return reachability.isReachable
    }

    /// Get the App Version
    static func getVersion() -> String {
        return Bundle.main.infoDictionary?["CFBundleShortVersionString"] as? String ?? "1"
    }

    /// Get the build version of app
    static func getBuildVersion() -> String {
        return Bundle.main.infoDictionary?["CFBundleVersion"] as? String ?? "1"
    }

    /// show banner Failure message.
    static func showErrorMessage(strMessage: String) {
        DispatchQueue.main.async {
            BannerAlert().showBannerMessage(message: strMessage,
                                            type: BannerMessageType.failure.rawValue,
                                            showDismissBtn: ShowButton.hide.rawValue)
        }
    }

    /// show banner Warning message.
    static func showWarningMessage(strMessage: String) {
        DispatchQueue.main.async {
            BannerAlert().showBannerMessage(message: strMessage,
                                            type: BannerMessageType.warning.rawValue,
                                            showDismissBtn: ShowButton.hide.rawValue)
        }
    }

    /// Show banner Success message.
    static func showSuccessMessage(strMessage: String) {
        DispatchQueue.main.async {
            BannerAlert().showBannerMessage(message: strMessage,
                                            type: BannerMessageType.success.rawValue,
                                            showDismissBtn: ShowButton.hide.rawValue)
        }
    }

    /// Show Progress Loader.
    static func showProgressHUD(view: UIView, shouldShowOnlyLoader: Bool = false) {
        DispatchQueue.main.async {
            let showHUD = MBProgressHUD.showAdded(to: view, animated: true)
            if !shouldShowOnlyLoader {
                showHUD.label.numberOfLines = 2
                showHUD.label.text = "Please wait"
                showHUD.detailsLabel.text = "Loading..."
                showHUD.bezelView.color = UIColor.primaryBlack.withAlphaComponent(0.7)
                showHUD.contentColor = .white
            } else {
                showHUD.label.numberOfLines = 0
                showHUD.label.text = ""
                showHUD.detailsLabel.text = ""
                showHUD.bezelView.color = .clear
                showHUD.contentColor = .primaryBlack
            }
            showHUD.animationType = .zoomOut
            showHUD.bezelView.style = .solidColor
            showHUD.label.textColor = .white
            showHUD.detailsLabel.textColor = .white
        }
    }

    /// Hide Progress Loader.
    static func hideProgressHUD(view: UIView) {
        DispatchQueue.main.async {
            MBProgressHUD.hide(for: view, animated: true)
        }
    }

    static func isValidEmail(_ email: String) -> Bool {
        let emailRegex = "[A-Z0-9a-z._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,4}"
        let emailPredicate = NSPredicate(format: "self matches %@", emailRegex)
        return emailPredicate.evaluate(with: email)
    }

    static func getAttributedString(textAlignment: NSTextAlignment,
                                    message: String,
                                    lineSpacing: CGFloat = 1,
                                    messageFont: UIFont,
                                    messageTextColor: UIColor,
                                    arrAttributes: [AttributedString]) -> NSMutableAttributedString
    {
        let paragraphStyle = NSMutableParagraphStyle()
        paragraphStyle.lineSpacing = lineSpacing
        paragraphStyle.alignment = textAlignment

        let mutableStringTitle = NSMutableAttributedString(string: message,
                                                           attributes: [NSAttributedString.Key.font: messageFont,
                                                                        NSAttributedString.Key.paragraphStyle: paragraphStyle,
                                                                        NSAttributedString.Key.foregroundColor: messageTextColor])

        for index in 0 ..< arrAttributes.count {
            if let range = message.range(of: arrAttributes[index].subString) {
                if arrAttributes[index].attributeType != nil {
                    mutableStringTitle.addAttribute(arrAttributes[index].attributeType,
                                                    value: arrAttributes[index].subString,
                                                    range: NSRange(range, in: message))
                }

                let startPos = message.distance(from: message.startIndex, to: range.lowerBound)

                if arrAttributes[index].isUnderlined {
                    mutableStringTitle.addAttributes([NSAttributedString.Key.font: arrAttributes[index].subStringFont,
                                                      NSAttributedString.Key.foregroundColor: arrAttributes[index].subStringColor,
                                                      NSAttributedString.Key.underlineStyle: arrAttributes[index].underlineStyle,
                                                      NSAttributedString.Key.underlineColor: arrAttributes[index].underlineColor],
                                                     range: NSRange(location: startPos, length: arrAttributes[index].subString.count))
                } else if arrAttributes[index].isStrikeThrough {
                    mutableStringTitle.addAttributes([NSAttributedString.Key.font: arrAttributes[index].subStringFont,
                                                      NSAttributedString.Key.foregroundColor: arrAttributes[index].subStringColor,
                                                      NSAttributedString.Key.strikethroughStyle: NSUnderlineStyle.single.rawValue],
                                                     range: NSRange(location: startPos, length: arrAttributes[index].subString.count))
                } else {
                    mutableStringTitle.addAttributes([NSAttributedString.Key.font: arrAttributes[index].subStringFont,
                                                      NSAttributedString.Key.foregroundColor: arrAttributes[index].subStringColor],
                                                     range: NSRange(location: startPos, length: arrAttributes[index].subString.count))
                }
                mutableStringTitle.addAttribute(NSAttributedString.Key.backgroundColor, value: arrAttributes[index].backgroundColor, range: NSRange(location: startPos, length: arrAttributes[index].subString.count))
            }
        }

        return mutableStringTitle
    }

    // Get the formatted price [currency = $]
    static func getCurrencyFormatWithComma(price: Double) -> String {
        let formatter = NumberFormatter()
        formatter.locale = Locale(identifier: "en_US")
        formatter.roundingMode = .halfUp
        formatter.numberStyle = .currency
        formatter.minimumFractionDigits = 1
        formatter.maximumFractionDigits = 2
        formatter.currencySymbol = "€"
        formatter.currencyGroupingSeparator = ","
        formatter.minimumIntegerDigits = 1
        let str = formatter.string(from: NSNumber(value: price)) ?? ""
        return str
    }

    static func getBoolValFromInt(value: Int) -> Bool {
        switch value {
        case 1:
            return true
        default:
            return false
        }
    }

    static func refreshProfileInfo() {
        NotificationCenter.default.post(name: .refreshProfileInfo, object: nil)
    }

    static func detectUserCountry() -> String? {
        guard let countryCode = Locale.current.regionCode else {
            return nil
        }

        return countryCode
    }

    static func clearPreferenceData() {
        AccountRepository.clearPreferenceData()
    }

    // MARK: - DB operations 
    static func addCustomer(){
        let dateFormatter = DateFormatter()
        dateFormatter.dateFormat = "dd/MM/yyyy"
        let currentDate = dateFormatter.string(from: Date())

        let dataBase = DatabaseManager.shared
        dataBase.insertCustomerItem(item: Customer(id: 1, email: "v.gryparis@amco.gr", datetime: currentDate))
        dataBase.insertCustomerItem(item: Customer(id: 2, email: "c.magalios@amco.gr", datetime: currentDate))
        dataBase.insertCustomerItem(item: Customer(id: 3, email: "vkakavas@amco.gr", datetime: currentDate))
        dataBase.insertCustomerItem(item: Customer(id: 4, email: "e.gavrielatou@amco.gr", datetime: currentDate))
    }

    static func addRoute(){
        let databaseHelper = DatabaseManager.shared
        databaseHelper.insertBusRouteItem(item: BusRoute.init(routeId: 1, routeCode: "001", routeValue: "01-N.SMURNI-NEAPOLI"))
        databaseHelper.insertBusRouteItem(item: BusRoute.init(routeId: 2, routeCode: "006", routeValue: "02-K. PLATEIA - TERMA BOLOU"))
        databaseHelper.insertBusRouteItem(item: BusRoute.init(routeId: 3, routeCode: "014", routeValue: "03-PRAKTIKER - ALKAZAR"))
    }

    static func addBusStop(){
        let databaseHelper = DatabaseManager.shared
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 1, stopCode: "0101", stopValue: "NEA SMYRNH AFETHRIA", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 2, stopCode: "0102", stopValue: "EMPORIKO KENTRO- PERIFEREIA", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 3, stopCode: "0103", stopValue: "GALAKTOS", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 4, stopCode: "0104", stopValue: "NOSTOS", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 5, stopCode: "0105", stopValue: "ERGATIKES KATOIKIES NEAS SMYRNHS", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 6, stopCode: "0106", stopValue: "AGIOS DHMHTRIOS", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 7, stopCode: "0107", stopValue: "PALIO FALHRO", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 8, stopCode: "0108", stopValue: "IKA NEAS SMYRNHS", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 9, stopCode: "0109", stopValue: "KEP- SFAGEIA", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 10, stopCode: "0110", stopValue: "START OF 31ST AUGUST", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 11, stopCode: "0111", stopValue: "GENIKO NOSOKOMEIO LARISAS", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 12, stopCode: "0112", stopValue: "ODEIO", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 13, stopCode: "0113", stopValue: "OTE", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 14, stopCode: "0114", stopValue: "DIKASTIKO MEGARO", stopRouteCode: "001"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 15, stopCode: "0143", stopValue: "KENTRIKH PLATEIA", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 16, stopCode: "0202", stopValue: "FISHERIES", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 17, stopCode: "0203", stopValue: "PARKO AGIOU KWNSTANTINOU", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 18, stopCode: "0204", stopValue: "IPSILANTOU- HPEIROU", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 19, stopCode: "0205", stopValue: "TERMA YPSHLANTOU", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 20, stopCode: "0206", stopValue: "EUROBANK", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 21, stopCode: "0207", stopValue: "GRAMMES VOLOU", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 22, stopCode: "0208", stopValue: "S/M LIDL", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 23, stopCode: "0209", stopValue: "S/M GALAXIAS", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 24, stopCode: "0210", stopValue: "NIAKAS", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 25, stopCode: "0211", stopValue: "S/M TSOUMANHS", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 26, stopCode: "0212", stopValue: "GIANNOUKA", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 27, stopCode: "0213", stopValue: "FALHRO", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 28, stopCode: "0214", stopValue: "STATHMOS OTE", stopRouteCode: "006"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 29, stopCode: "0329", stopValue: "PRAKTIKER", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 30, stopCode: "0330", stopValue: "BIOKARPET", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 31, stopCode: "0331", stopValue: "XELIDWNH", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 32, stopCode: "0332", stopValue: "STATISTIKHS", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 33, stopCode: "0333", stopValue: "IMPERIAL", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 34, stopCode: "0334", stopValue: "PLATEIA ABERWF", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 35, stopCode: "0335", stopValue: "XALATSIS", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 36, stopCode: "0336", stopValue: "FAROS", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 37, stopCode: "0337", stopValue: "S/M LIDL", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 38, stopCode: "0338", stopValue: "POLYKATOIKIES", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 39, stopCode: "0339", stopValue: "STRATOPEDO MPOUGA", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 40, stopCode: "0340", stopValue: "SARAKAKIS", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 41, stopCode: "0341", stopValue: "S M MY MARKET", stopRouteCode: "014"))
        databaseHelper.insertBusStopItem(item: BusStops.init(stopId: 42, stopCode: "0342", stopValue: "KALH KARDIA", stopRouteCode: "014"))
        
    }
}
