import UIKit
import SwiftUI
import UIKit

// swiftlint:disable:next avoid_UIViewController
extension UIViewController {
    /// Hides keyboard
    func hideKeyboard() {
        view.endEditing(true)
    }

    /// Performs segue with identifier and sets blank navigation back text.
    ///
    /// - Parameters:
    ///   - withIdentifier: Segue identifier
    ///   - sender: Sender description
    ///   - removeNavBackTitle: Whether to set blank navigation back text or not.
    func performSegue(withIdentifier: String, sender: Any, removeNavBackTitle: Bool) {
        if removeNavBackTitle {
            navigationItem.backBarButtonItem = UIBarButtonItem(title: "", style: UIBarButtonItem.Style.plain, target: nil, action: nil)
        }
        performSegue(withIdentifier: withIdentifier, sender: sender)
    }

    /// Adds child view controller to the parent.
    /// 
    /// - Parameter child: Child view controller.
    // swiftlint:disable:next avoid_UIViewController
    func add(_ child: UIViewController) {
        addChild(child)
        view.addSubview(child.view)
        child.didMove(toParent: self)
    }

    /// It removes the child view controller from the parent.
    func remove() {
        guard parent != nil else {
            return
        }
        willMove(toParent: nil)
        removeFromParent()
        view.removeFromSuperview()
    }

    // swiftlint:disable:next avoid_UIViewController
    var previousViewController: UIViewController? {
        if let controllersOnNavStack = navigationController?.viewControllers, controllersOnNavStack.count >= 2 {
            return controllersOnNavStack[controllersOnNavStack.count - 2]
        }
        return nil
    }

    func hideKeyboardWhenTappedAround() {
        // swiftlint:disable:next avoid_UIViewController
        let tap = UITapGestureRecognizer(target: self, action: #selector(UIViewController.dismissKeyboard))
        tap.cancelsTouchesInView = false
        view.addGestureRecognizer(tap)
    }

    @objc func dismissKeyboard() {
        view.endEditing(true)
    }

    func openSettings() {
        let settingsAppURL = URL(string: UIApplication.openSettingsURLString)!
        UIApplication.shared.open(settingsAppURL,
                                  options: [:],
                                  completionHandler: nil)
    }

    func present<Content: View>(style: UIModalPresentationStyle = .automatic, transitionStyle: UIModalTransitionStyle = .coverVertical, @ViewBuilder builder: () -> Content) {
        let toPresent = UIHostingController(rootView: AnyView(EmptyView()))
        toPresent.modalPresentationStyle = style
        toPresent.modalTransitionStyle = transitionStyle
        toPresent.view.backgroundColor = UIColor.init(red: 0, green: 0, blue: 0, alpha: 0.5)
        toPresent.rootView = AnyView(
            builder()
                .environment(\.viewController, toPresent)
        )
        self.present(toPresent, animated: true, completion: nil)
    }
    func showCustomDialog(title: String,
                          content: String = "",
                          attributedContent: NSMutableAttributedString? = nil,
                          negativeButtonText: String = "",
                          onTapNegativeButton: (() -> (()))? = nil,
                          neutralButtonText: String = "",
                          onTapNeutralButton: (() -> (()))? = nil,
                          positiveButtonText: String = "",
                          onTapPositiveButton: (() -> (()))? = nil)
    {
        let customDialogViewModel = CustomDialogViewModel(
            title: title,
            content: content,
            attributedContent: attributedContent,
            negativeButtonText: negativeButtonText,
            onTapNegativeButton: onTapNegativeButton,
            neutralButtonText: neutralButtonText,
            onTapNeutralButton: onTapNeutralButton,
            positiveButtonText: positiveButtonText,
            onTapPositiveButton: onTapPositiveButton)
        let dialog = CustomDialog(viewModel: customDialogViewModel)
        dialog.modalPresentationStyle = .overFullScreen
        dialog.modalTransitionStyle = .crossDissolve
        present(dialog, animated: true, completion: nil)
    }
}
