//
//  UIImageExt.swift
//

import ImageIO
import UIKit

extension UIImage {
    func resizeImage(image: UIImage, targetSize: CGSize) -> UIImage {
        let size = image.size
        let widthRatio = targetSize.width / size.width
        let heightRatio = targetSize.height / size.height
        // Figure out what our orientation is, and use that to form the rectangle
        var newSize: CGSize
        if widthRatio > heightRatio {
            newSize = CGSize(width: size.width * heightRatio, height: size.height * heightRatio)
        } else {
            newSize = CGSize(width: size.width * widthRatio, height: size.height * widthRatio)
        }
        // This is the rect that we've calculated out and this is what is actually used below
        let rect = CGRect(x: 0, y: 0, width: newSize.width, height: newSize.height)
        // Actually do the resizing to the rect using the ImageContext stuff
        UIGraphicsBeginImageContextWithOptions(newSize, false, 1.0)
        image.draw(in: rect)
        let newImage = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        return newImage!
    }

    func resized(withPercentage percentage: CGFloat) -> UIImage? {
        let canvasSize = CGSize(width: size.width * percentage, height: size.height * percentage)
        UIGraphicsBeginImageContextWithOptions(canvasSize, false, scale)
        defer { UIGraphicsEndImageContext() }
        draw(in: CGRect(origin: .zero, size: canvasSize))
        return UIGraphicsGetImageFromCurrentImageContext()
    }

    func resized(toWidth width: CGFloat) -> UIImage? {
        let canvasSize = CGSize(width: width, height: CGFloat(ceil(width / size.width * size.height)))
        UIGraphicsBeginImageContextWithOptions(canvasSize, false, scale)
        defer { UIGraphicsEndImageContext() }
        draw(in: CGRect(origin: .zero, size: canvasSize))
        return UIGraphicsGetImageFromCurrentImageContext()
    }

    func resizeJPEGByByte(maxByte: Int) {
        var compressQuality: CGFloat = 1
        var imageByte = jpegData(compressionQuality: 1)?.count
        while imageByte! > maxByte {
            imageByte = jpegData(compressionQuality: compressQuality)?.count
            compressQuality -= 0.1
        }
    }

    func resizePNGByByte(maxByte: Int) {
        var compressQuality: CGFloat = 1
        var imageByte = pngData()?.count
        while imageByte! > maxByte {
            imageByte = resized(withPercentage: compressQuality)!.pngData()?.count
            compressQuality -= 0.1
        }
    }

    func resizeByByte(maxByte: Int) -> Data {
        var compressQuality: CGFloat = 1
        var imageByte = 0
        var fileType = 1 // 1 =  JPEG 2 = PNG
        var imgData: Data?
        if let data: Data = jpegData(compressionQuality: 1) {
            fileType = 1
            imageByte = data.count
            imgData = jpegData(compressionQuality: compressQuality)!
        } else if let data: Data = pngData() {
            fileType = 2
            imageByte = data.count
            imgData = resized(withPercentage: compressQuality)!.pngData()!
        }
        while imageByte > maxByte {
            if fileType == 1 {
                imageByte = (jpegData(compressionQuality: compressQuality)?.count)!
                imgData = jpegData(compressionQuality: compressQuality)!
            } else {
                imageByte = (resized(withPercentage: compressQuality)!.pngData()?.count)!
                imgData = resized(withPercentage: compressQuality)!.pngData()!
            }
            debugPrint(imageByte)
            debugPrint(compressQuality)
            compressQuality -= 0.1
        }
        return imgData!
    }

//    func resizeByByte(maxByte: Int, keyName: String) -> MultiPartData {
//        var compressQuality: CGFloat = 0.5
//        var imageByte: Int = 0
//        var fileType: Int = 1 // 1 =  JPEG 2 = PNG
//        var imgData: Data?
//        var multiPartData: MultiPartData?
//        if let data: Data = self.jpegData(compressionQuality: 1) {
//            fileType = 1
//            imageByte = data.count
//            imgData = self.jpegData(compressionQuality: compressQuality)!
//        }
//        else if let data: Data = self.pngData() {
//            fileType = 2
//            imageByte = data.count
//            imgData = resized(withPercentage: compressQuality)!.pngData()!
//        }
//        while imageByte > maxByte {
//            if fileType == 1 {
//                imageByte = (self.jpegData(compressionQuality: compressQuality)?.count)!
//                imgData = self.jpegData(compressionQuality: compressQuality)!
//            }
//            else {
//                imageByte = (resized(withPercentage: compressQuality)!.pngData()?.count)!
//                imgData = resized(withPercentage: compressQuality)!.pngData()!
//            }
//            debugPrint(imageByte)
//            debugPrint(compressQuality)
//            compressQuality -= 0.1
//        }
//        if fileType == 1 {
//            multiPartData = MultiPartData(data: imgData!, fileName: String(Int(Date().timeIntervalSince1970 * 1000))+".jpeg", keyName: keyName, mimeType: "image/jpeg")
//        }
//        else {
//            multiPartData = MultiPartData(data: imgData!, fileName: String(Int(Date().timeIntervalSince1970 * 1000))+".png", keyName: keyName, mimeType: "image/png")
//        }
//        return multiPartData!
//    }
    static func drawDottedImage(width: CGFloat, height: CGFloat, color: UIColor) -> UIImage {
        let path = UIBezierPath()
        path.move(to: CGPoint(x: 1.0, y: 1.0))
        path.addLine(to: CGPoint(x: width, y: 1))
        path.lineWidth = 1.5
        let dashes: [CGFloat] = [path.lineWidth, path.lineWidth * 5]
        path.setLineDash(dashes, count: 2, phase: 0)
        path.lineCapStyle = .butt
        UIGraphicsBeginImageContextWithOptions(CGSize(width: width, height: height), false, 2)
        color.setStroke()
        path.stroke()
        let image: UIImage = UIGraphicsGetImageFromCurrentImageContext()!
        UIGraphicsEndImageContext()
        return image
    }

    func getImage(image: UIImage, backgroundColor: UIColor) -> UIImage? {
        UIGraphicsBeginImageContextWithOptions(image.size, false, image.scale)
        backgroundColor.setFill()
        // Image with  normal Background color
        UIRectFill(CGRect(origin: .zero, size: image.size))
        // Image with Circular Background color
        //        let rect = CGRect(origin: .zero, size: image.size)
        //        let path = UIBezierPath(arcCenter: CGPoint(x:rect.midX, y:rect.midY), radius: rect.midX, startAngle: 0, endAngle: 6.28319, clockwise: true)
        //        path.fill()
        image.draw(at: .zero)
        let newImage = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        return newImage
    }

    public class func gif(data: Data) -> UIImage? {
        // Create source from data
        guard let source = CGImageSourceCreateWithData(data as CFData, nil) else {
            debugPrint("SwiftGif: Source for the image does not exist")
            return nil
        }
        return UIImage.animatedImageWithSource(source)
    }

    public class func gif(url: String) -> UIImage? {
        // Validate URL
        guard let bundleURL = URL(string: url) else {
            debugPrint("SwiftGif: This image named \"\(url)\" does not exist")
            return nil
        }
        // Validate data
        guard let imageData = try? Data(contentsOf: bundleURL) else {
            debugPrint("SwiftGif: Cannot turn image named \"\(url)\" into NSData")
            return nil
        }
        return gif(data: imageData)
    }

    public class func gif(name: String) -> UIImage? {
        // Check for existance of gif
        guard let bundleURL = Bundle.main
            .url(forResource: name, withExtension: "gif") else
        {
            debugPrint("SwiftGif: This image named \"\(name)\" does not exist")
            return nil
        }
        // Validate data
        guard let imageData = try? Data(contentsOf: bundleURL) else {
            debugPrint("SwiftGif: Cannot turn image named \"\(name)\" into NSData")
            return nil
        }
        return gif(data: imageData)
    }

    @available(iOS 9.0, *)
    public class func gif(asset: String) -> UIImage? {
        // Create source from assets catalog
        guard let dataAsset = NSDataAsset(name: asset) else {
            debugPrint("SwiftGif: Cannot turn image named \"\(asset)\" into NSDataAsset")
            return nil
        }
        return gif(data: dataAsset.data)
    }

    internal class func delayForImageAtIndex(_ index: Int, source: CGImageSource!) -> Double {
        var delay = 0.1
        // Get dictionaries
        let cfProperties = CGImageSourceCopyPropertiesAtIndex(source, index, nil)
        let gifPropertiesPointer = UnsafeMutablePointer<UnsafeRawPointer?>.allocate(capacity: 0)
        defer {
            gifPropertiesPointer.deallocate()
        }
        if CFDictionaryGetValueIfPresent(cfProperties, Unmanaged.passUnretained(kCGImagePropertyGIFDictionary).toOpaque(), gifPropertiesPointer) == false {
            return delay
        }
        let gifProperties: CFDictionary = unsafeBitCast(gifPropertiesPointer.pointee, to: CFDictionary.self)
        // Get delay time
        var delayObject: AnyObject = unsafeBitCast(
            CFDictionaryGetValue(gifProperties,
                                 Unmanaged.passUnretained(kCGImagePropertyGIFUnclampedDelayTime).toOpaque()),
            to: AnyObject.self
        )
        if delayObject.doubleValue == 0 {
            delayObject = unsafeBitCast(CFDictionaryGetValue(gifProperties,
                                                             Unmanaged.passUnretained(kCGImagePropertyGIFDelayTime).toOpaque()), to: AnyObject.self)
        }
        if let delayObject = delayObject as? Double, delayObject > 0 {
            delay = delayObject
        } else {
            delay = 0.1 // Make sure they're not too fast
        }
        return delay
    }

    internal class func gcdForPair(alpha: Int?, beta: Int?) -> Int {
        var alpha = alpha
        var beta = beta
        // Check if one of them is nil
        if beta == nil || alpha == nil {
            if beta != nil {
                return beta!
            } else if alpha != nil {
                return alpha!
            } else {
                return 0
            }
        }
        // Swap for modulo
        if alpha! < beta! {
            let cat = alpha
            alpha = beta
            beta = cat
        }
        // Get greatest common divisor
        var rest: Int
        while true {
            rest = alpha! % beta!
            if rest == 0 {
                return beta! // Found it
            } else {
                alpha = beta
                beta = rest
            }
        }
    }

    internal class func gcdForArray(_ array: [Int]) -> Int {
        if array.isEmpty {
            return 1
        }
        var gcd = array[0]
        for val in array {
            gcd = UIImage.gcdForPair(alpha: val, beta: gcd)
        }
        return gcd
    }

    internal class func animatedImageWithSource(_ source: CGImageSource) -> UIImage? {
        let count = CGImageSourceGetCount(source)
        var images = [CGImage]()
        var delays = [Int]()
        // Fill arrays
        for index in 0 ..< count {
            // Add image
            if let image = CGImageSourceCreateImageAtIndex(source, index, nil) {
                images.append(image)
            }
            // At it's delay in cs
            let delaySeconds = UIImage.delayForImageAtIndex(Int(index),
                                                            source: source)
            delays.append(Int(delaySeconds * 1000.0)) // Seconds to ms
        }
        // Calculate full duration
        let duration: Int = {
            var sum = 0
            for val: Int in delays {
                sum += val
            }
            return sum
        }()
        // Get frames
        let gcd = gcdForArray(delays)
        var frames = [UIImage]()
        var frame: UIImage
        var frameCount: Int
        for index in 0 ..< count {
            frame = UIImage(cgImage: images[Int(index)])
            frameCount = Int(delays[Int(index)] / gcd)
            for _ in 0 ..< frameCount {
                frames.append(frame)
            }
        }
        let animation = UIImage.animatedImage(with: frames,
                                              duration: Double(duration) / 1000.0)
        return animation
    }

    func cropImage(toRect rect: CGRect) -> UIImage {
        let imageRef: CGImage = cgImage!.cropping(to: rect)!
        let croppedImage = UIImage(cgImage: imageRef)
        return croppedImage
    }
}
