//
//  DataExt.swift
//  BeInBeOutAMCO
//
//  Created by My on 01/09/23.
//

import Foundation

extension Data {

    init<T>(from value: T) {
        self = Swift.withUnsafeBytes(of: value) { Data($0) }
    }
    var hexString: String {
        return map { String(format: "%02hhx", $0) }.joined()
    }
    func to<T>(type: T.Type) -> T? where T: ExpressibleByIntegerLiteral {
        var value: T = 0
        guard count >= MemoryLayout.size(ofValue: value) else { return nil }
        _ = Swift.withUnsafeMutableBytes(of: &value, { copyBytes(to: $0)} )
        return value
    }

    struct HexEncodingOptions: OptionSet {
        let rawValue: Int
        static let upperCase = HexEncodingOptions(rawValue: 1 << 0)
    }

    func hexEncodedString(options: HexEncodingOptions = []) -> String {
        let format = options.contains(.upperCase) ? "%02hhX" : "%02hhx"
        return map { String(format: format, $0) }.joined()
    }

    init?(fromHexEncodedString string: String) {

        // Convert 0 ... 9, a ... f, A ...F to their decimal value,
        // return nil for all other input characters
        func decodeNibble(uint: UInt16) -> UInt8? {
            switch(uint) {
            case 0x30 ... 0x39:
                return UInt8(uint - 0x30)
            case 0x41 ... 0x46:
                return UInt8(uint - 0x41 + 10)
            case 0x61 ... 0x66:
                return UInt8(uint - 0x61 + 10)
            default:
                return nil
            }
        }

        self.init(capacity: string.utf16.count/2)
        var even = true
        var byte: UInt8 = 0
        for cunit in string.utf16 {
            guard let val = decodeNibble(uint: cunit) else { return nil }
            if even {
                byte = val << 4
            } else {
                byte += val
                self.append(byte)
            }
            even = !even
        }
        guard even else { return nil }
    }

    enum Endianess {
        case little
        case big
    }

    func toFloat(endianess: Endianess = .little) -> Float? {
        guard self.count <= 4 else { return nil }

        switch endianess {
        case .big:
            let data = [UInt8](repeating: 0x00, count: 4-self.count) + self
            return data.withUnsafeBytes { $0.load(as: Float.self) }
        case .little:
            let data = self + [UInt8](repeating: 0x00, count: 4-self.count)
            return data.reversed().withUnsafeBytes { $0.load(as: Float.self) }
        }
    }
}
