//
//  HomeScreenViewController.swift
//  BeInBeOutAMCO
//
//  Created by Mac-01 on 01/09/23.
//

import CoreBluetooth
import UIKit

struct BluetoothDeviceData {
    var peripheralDevice: CBPeripheral
    var advertisementData: [String: Any]
    var rssi: NSNumber
}

class HomeScreenViewController: BaseHostingViewController<HomeView> {
    // MARK: - Properties

    private var viewModel: HomeViewModel
    var availableDevices: [BluetoothDeviceData] = []
    var centralManager: CBCentralManager!
    var peripheralDevice: CBPeripheral!
    var isScannig = false
    var isBtOff = false
    var isBTAvailable = false
    var isPopupShow = false
    var isRangeOutPopupShow = false
    var routeName: String = ""
    var nextStationName: String = ""
    var leftStationName: String = ""
    var onBoardStationName: String = ""
    var onBoardStationCode: String = ""
    var tempUUID = ""
//    var IS_ONCE_OPEN_IN: Bool = false
//    var IS_ONCE_OPEN_OUT: Bool = false

    // MARK: - initView

    init(viewModel: HomeViewModel) {
        self.viewModel = viewModel
        super.init(with: HomeView(viewModel: viewModel))
        self.centralManager = CBCentralManager(delegate: self, queue: nil)
    }

    @available(*, unavailable)
    required init?(coder _: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }

    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        navigationController?.setNavigationBarHidden(true, animated: false)
        updateWallet()
    }

    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        navigationController?.setNavigationBarHidden(false, animated: false)
    }

    // MARK: - Setup Views

    override func setupViews() {
        super.setupViews()
        self.viewModel.uuid = self.viewModel.getRandomNumberString()
        self.setupActions()
    }

    // MARK: - Actions

    private func setupActions() {}

    func scanBleDevices() {
        if self.isScannig {
            self.centralManager.stopScan()
            self.isScannig = false
        } else {
            self.centralManager.stopScan()
            self.availableDevices.removeAll()
            let options: [String: Any] = [CBCentralManagerScanOptionAllowDuplicatesKey: false]
            self.centralManager.scanForPeripherals(withServices: nil, options: options)
            if self.isBTAvailable {
                self.isScannig = true

            } else {
//                let alerview = UIAlertController(title: btStatus, message: nil, preferredStyle: .alert)
//                let actionOK = UIAlertAction(title: "OK", style: .default)
//                alerview.addAction(actionOK)
//                present(alerview, animated: true)
            }
        }
    }
}

// MARK: - BLE Functinalities

extension HomeScreenViewController: CBCentralManagerDelegate, CBPeripheralDelegate {
    func centralManagerDidUpdateState(_ central: CBCentralManager) {
        switch central.state {
        case .unknown:
            debugPrint("central.state is .unknown")
            self.isBTAvailable = false

        case .resetting:
            debugPrint("central.state is .resetting")
            self.isBTAvailable = false

        case .unsupported:
            debugPrint("central.state is .unsupported")
            self.isBTAvailable = false

        case .unauthorized:
            debugPrint("central.state is .unauthorized")
            self.isBTAvailable = false

            self.isScannig = false

        case .poweredOff:
            debugPrint("central.state is .poweredOff")
            self.isBTAvailable = false

            if self.isScannig {
                self.isScannig = false
                self.isBtOff = true
                self.tempUUID = self.viewModel.getRandomNumberString()
                self.viewModel.getEvent(status: "BLUETOOTH_CON_ALERT", uuid: self.tempUUID) { result in

                    switch result {
                    case .success(let response):
                        self.hostingController.rootView.eWalletAmt = Helper.getCurrencyFormatWithComma(price: (response.walletBalance > 0) ? response.walletBalance / 100 : 0)

                    case .failure:
                        break
                    }
                    self.present(style: .overCurrentContext, transitionStyle: .crossDissolve) {
                        CustomPopUp(title: "alert".localized,
                                    message: "turn_off".localized, attributedString: nil, leadingButtontitle: "",
                                    trailingButtontitle: "open_setting".localized,
                                    leadingButtonCallback: {}, trailingButtonCallback: {
                                        self.centralManager = CBCentralManager(delegate: self, queue: nil)
                                    })
                    }
                }
            }

        case .poweredOn:
            debugPrint("central.state is .poweredOn")
            self.isBTAvailable = true
            self.isScannig = false
            if self.isBtOff {
                self.isBtOff = false
                self.viewModel.getEvent(status: "BLUETOOTH_CON_ALERT_CLEAR", uuid: self.tempUUID) { result in

                    switch result {
                    case .success(let response):
                        self.hostingController.rootView.eWalletAmt = Helper.getCurrencyFormatWithComma(price: (response.walletBalance > 0) ? response.walletBalance / 100 : 0)

                    case .failure:
                        break
                    }
                }
            }
            self.scanBleDevices()

        @unknown default:
            debugPrint("central.state is .default")
            self.isBTAvailable = false
        }
    }

    func estimateDistance(rssi: Double, txPower: Double = -59, pathLossExponent: Double = 2.0) -> Double {
        let ratio = rssi - txPower
        let distance = pow(10, ratio / (10 * pathLossExponent))
        return distance
    }

    func centralManager(_: CBCentralManager, didDiscover peripheral: CBPeripheral, advertisementData: [String: Any], rssi RSSI: NSNumber) {
        if let manufacturerData = advertisementData[CBAdvertisementDataManufacturerDataKey] {
            let data = manufacturerData as! NSData

            if data.length == 25 {
                if let manufacturerData1 = advertisementData[CBAdvertisementDataManufacturerDataKey] as? Data {
                    guard data.count >= 21 else {
                        debugPrint("Manufacturer data is too short")
                        return
                    }
                    self.availableDevices.removeAll()
                    let identifier = manufacturerData1[0..<4].hexString

                    if identifier == "5900beac" {
                        let identifier1 = manufacturerData1[4..<manufacturerData1.count - 5].hexString

                        let identifier2 = manufacturerData1[manufacturerData1.count - 5..<manufacturerData1.count - 3].hexString

                        let identifier3 = manufacturerData1[manufacturerData1.count - 3..<manufacturerData1.count - 1].hexString

                        let hexatoAscii = identifier1.decodeHex() ?? ""

                        let serialNo = String(identifier1.prefix(4))
                        let role = String(identifier1.prefix(6).suffix(2))
                        let extra = String(hexatoAscii.prefix(3))
                        let operatorId = String(hexatoAscii.prefix(7).suffix(4))
                        self.viewModel.busID = String(hexatoAscii.prefix(10).suffix(3))
                        self.viewModel.routeId = String(hexatoAscii.prefix(14).suffix(4))
                        let stopCode1 = String(hexatoAscii.prefix(16).suffix(2))
                        if let decimalValue = Int(identifier2, radix: 16) {
                            // You now have the decimal value in decimalValue
                            // You can use decimalValue here
                            let stringHexa = String(decimalValue, radix: 16)
                            let stopCode2 = stopCode1 + (stringHexa.decodeHex() ?? "")
                            self.viewModel.stopID = stopCode2
                            self.leftStationName = stopCode2
                        } else {
                            // Handle the case where hexString is not a valid hexadecimal string
                            debugPrint("Invalid Hexadecimal String")
                        }
                        let device = BluetoothDeviceData(peripheralDevice: peripheral, advertisementData: advertisementData, rssi: RSSI)
                        self.availableDevices.append(device)

                        self.isBTAvailable = true
                    }
                }
                self.manageRangeOfDevice()
            }
        }
    }

    func updateWallet() {
        self.viewModel.getEvents { result in
            switch result {
            case .success(let response):
                Helper.hideProgressHUD(view: self.view)
                AccountRepository().setWallet(wallet: (response.data.walletBalance > 0) ? response.data.walletBalance / 100 : 0)
                self.hostingController.rootView.eWalletAmt = Helper.getCurrencyFormatWithComma(price: (response.data.walletBalance > 0) ? response.data.walletBalance / 100 : 0)
                self.hostingController.rootView.dailyCapping = Helper.getCurrencyFormatWithComma(price: (response.data.dailyFairCapping > 0) ? response.data.dailyFairCapping / 100 : 0)
                self.hostingController.rootView.moneySpend = Helper.getCurrencyFormatWithComma(price: (response.data.dailyUntilNow > 0) ? response.data.dailyUntilNow / 100 : 0)

            case .failure:
                Helper.hideProgressHUD(view: self.view)
            }
        }
    }

    func manageRangeOfDevice() {
        if !self.availableDevices.isEmpty {
            let deviceInfo = self.availableDevices[0]
            let distanceInMeters = pow(10, (-56 - deviceInfo.rssi.doubleValue) / (10 * 2))
            if distanceInMeters <= 25 {
                self.updateViewValues()
                self.updateAlertValue(status: "in")
            } else {
                self.updateAlertValue(status: "out")
            }
        }
    }

    func updateViewValues() {
        let routsArray = self.viewModel.getBusRoute(busRoute: self.viewModel.routeId.trim())

        if !routsArray.isEmpty {
            for item in routsArray {
                self.routeName = item.routeValue
                break
            }
        }
        let busStopArrayList = self.viewModel.getBusStops(routeId: self.viewModel.routeId.trim())
        if !busStopArrayList.isEmpty {
            for index in 0..<busStopArrayList.count {
                if busStopArrayList[index].stopCode == self.leftStationName.trim() {
                    self.nextStationName = busStopArrayList[index].stopValue
                    self.leftStationName = busStopArrayList[index].stopValue

                    if index == busStopArrayList.count - 1 {
                        self.onBoardStationName = ""
                        self.onBoardStationCode = ""
                    } else if index == 0 {
                        self.onBoardStationName = busStopArrayList[0].stopValue
                        self.onBoardStationCode = busStopArrayList[0].stopCode
                    } else if index == 1 {
                        self.onBoardStationName = busStopArrayList[0].stopValue
                        self.onBoardStationCode = busStopArrayList[0].stopCode
                    } else {
                        self.onBoardStationName = busStopArrayList[index - 1].stopValue
                        self.onBoardStationCode = busStopArrayList[index - 1].stopCode
                    }
                }
            }
        }

        self.viewModel.nextStop = self.nextStationName
        self.viewModel.currentStop = self.onBoardStationName
        self.viewModel.busLine = self.routeName
        self.hostingController.rootView.nextStop = self.nextStationName
        self.hostingController.rootView.currentStop = self.onBoardStationName
        self.hostingController.rootView.busLine = self.routeName
        debugPrint("nextStationName \(self.nextStationName)->\(self.onBoardStationName)->\(self.routeName)")

        let currentTime = Date()
        let formatter = DateFormatter()
        formatter.dateFormat = "dd/MM/yyyy HH:mm:ss"
        let dateString = formatter.string(from: currentTime)
        self.viewModel.currentDateTime = dateString
    }

    func updateAlertValue(status: String) {
        if !self.isScannig {
            return
        }
        if status == "in" {
            if !self.isPopupShow {
                self.isPopupShow = true

                let attributeStriOne = AttributedString(subString: self.viewModel.currentDateTime, isUnderline: false, subStringColor: .primaryBlack, subStringFont: UIFont(name: AppFont.FontType.medium.rawValue, size: 17)!)

                let attributeStriTwo = AttributedString(subString: self.viewModel.currentStop, isUnderline: false, subStringColor: .primaryBlack, subStringFont: UIFont(name: AppFont.FontType.medium.rawValue, size: 17)!)

                let attributeStrThree = AttributedString(subString: self.viewModel.busLine, isUnderline: false, subStringColor: .primaryBlack, subStringFont: UIFont(name: AppFont.FontType.medium.rawValue, size: 17)!)

                let alertMsg = String(format: "board_at".localized, self.viewModel.currentDateTime, self.viewModel.currentStop, self.viewModel.busLine)

                let attributeMsg = Helper.getAttributedString(textAlignment: .left, message: alertMsg, messageFont: UIFont(name: AppFont.FontType.regular.rawValue, size: 15)!, messageTextColor: .primaryBlack, arrAttributes: [attributeStriOne, attributeStriTwo, attributeStrThree])

                self.present(style: .overCurrentContext, transitionStyle: .crossDissolve) {
                    CustomPopUp(title: "",
                                message: "", attributedString: attributeMsg,
                                leadingButtontitle: "no".localized,
                                trailingButtontitle: "agree".localized,
                                leadingButtonCallback: {
                                    self.hostingController.rootView.qrImage = "bus_home"
                                }, trailingButtonCallback: {
                                    self.isRangeOutPopupShow = false
                                    self.hostingController.rootView.qrImage = "img_qr"
                                    AccountRepository().setUUID(uuiD: self.viewModel.getRandomNumberString())
                                    self.viewModel.getEvent(status: "in", uuid: AccountRepository().getUUID()) { result in
                                        self.hostingController.rootView.attributedString = attributeMsg
                                        switch result {
                                        case .success(let response):
                                            self.hostingController.rootView.eWalletAmt = Helper.getCurrencyFormatWithComma(price: (response.walletBalance > 0) ? response.walletBalance / 100 : 0)

                                        case .failure:
                                            break
                                        }
                                    }
                                })
                }
            }
        } else {
            if !self.isRangeOutPopupShow {
                self.isRangeOutPopupShow = true
                let currentTime = Date()
                let formatter = DateFormatter()
                formatter.dateFormat = "dd/MM/yyyy HH:mm:ss"
                let dateString = formatter.string(from: currentTime)
                self.viewModel.beOutTime = dateString

                let attributeStriOne = AttributedString(subString: self.viewModel.beOutTime, isUnderline: false, subStringColor: .primaryBlack, subStringFont: UIFont(name: AppFont.FontType.medium.rawValue, size: 17)!)

                let attributeStriTwo = AttributedString(subString: self.viewModel.currentStop, isUnderline: false, subStringColor: .primaryBlack, subStringFont: UIFont(name: AppFont.FontType.medium.rawValue, size: 17)!)

                let attributeStrThree = AttributedString(subString: self.viewModel.busLine, isUnderline: false, subStringColor: .primaryBlack, subStringFont: UIFont(name: AppFont.FontType.medium.rawValue, size: 17)!)

                let alertMsg = String(format: "you_left_time".localized, self.viewModel.beOutTime, self.viewModel.currentStop, self.viewModel.busLine)

                let attributeMsg = Helper.getAttributedString(textAlignment: .left, message: alertMsg, messageFont: UIFont(name: AppFont.FontType.regular.rawValue, size: 15)!, messageTextColor: .primaryBlack, arrAttributes: [attributeStriOne, attributeStriTwo, attributeStrThree])
                self.hostingController.rootView.attributedString = nil
                self.present(style: .overCurrentContext, transitionStyle: .crossDissolve) {
                    CustomPopUp(title: "",
                                message: "", attributedString: attributeMsg,
                                leadingButtontitle: "no".localized,
                                trailingButtontitle: "agree".localized,
                                leadingButtonCallback: {}, trailingButtonCallback: {
                                    self.hostingController.rootView.qrImage = "bus_home"
                                    self.isPopupShow = false
                                    self.viewModel.getEvent(status: "out", uuid: AccountRepository().getUUID()) { result in

                                        switch result {
                                        case .success(let response):
                                            self.isScannig = false
                                            self.hostingController.rootView.eWalletAmt = Helper.getCurrencyFormatWithComma(price: (response.walletBalance > 0) ? response.walletBalance / 100 : 0)
                                            let rideCompleteVc = RideCompleteViewController(viewModel: RideCompleteViewModel(amount: Helper.getCurrencyFormatWithComma(price: (response.totalAmount > 0) ? response.totalAmount / 100 : 0)))
                                            self.navigationController?.pushViewController(rideCompleteVc, animated: true)
                                        case .failure:
                                            break
                                        }
                                    }
                                })
                }
            }
        }
    }

    func calculateDistanceFromRSSI(rssiValue: Double) -> Double {
        let txPower = -56.0 // Reference signal strength at 1 meter (dBm)
        let nValue = 2.0 // Path loss exponent (depends on environment)

        // Calculate distance using the simplified path loss formula
        let ratio = (txPower - rssiValue) / (10 * nValue)
        let distance = pow(10.0, ratio)

        return distance
    }

    func centralManager(_: CBCentralManager, didConnect peripheral: CBPeripheral) {
        debugPrint("connected to peripheral : \(peripheral.name ?? "")")

//        peripheralDevice.discoverServices(nil)
    }

    func peripheral(_ peripheral: CBPeripheral, didDiscoverServices _: Error?) {
        guard let services = peripheral.services else { return }

        for service in services {
            debugPrint("service==>\(service)")
            debugPrint("service.uuid==>\(service.uuid)")
            debugPrint("service.characteristics==>\(String(describing: service.characteristics))")

            //            peripheral.discoverCharacteristics([service.uuid], for: service)
            peripheral.discoverCharacteristics(nil, for: service)

            //            peripheral.discoverCharacteristics([bloodPressureCharacteristicUUID], for: service)
        }
    }

    func peripheral(_: CBPeripheral, didDiscoverCharacteristicsFor service: CBService, error _: Error?) {
        for characteristic in service.characteristics! {
            debugPrint("characteristic\(characteristic)")
        }
    }

    func peripheral(_ peripheral: CBPeripheral, didWriteValueFor _: CBCharacteristic, error: Error?) {
        if let error = error {
            debugPrint("Get error while writing data : \(error)")
        } else {
            debugPrint("Write data successfully to peripheral \(peripheral.name ?? "")")
        }
    }

    func hexToDecimal(_ hexString: String) -> Int? {
        let hexDigits = "0123456789ABCDEF"
        let uppercaseHexString = hexString.uppercased()
        var decimalValue = 0

        for char in uppercaseHexString {
            if let digitValue = hexDigits.firstIndex(of: char) {
                decimalValue = decimalValue * 16 + hexDigits.distance(from: hexDigits.startIndex, to: digitValue)
            } else {
                // Invalid character in the input string
                return nil
            }
        }

        return decimalValue
    }
}
