//
//  ReachabilityError.swift
//

import Foundation
import SystemConfiguration

public enum ReachabilityError: Error {
    case failedToCreateWithAddress(sockaddr_in)
    case failedToCreateWithHostname(String)
    case unableToSetCallback
    case unableToSetDispatchQueue
}

public let reachabilityChangedNotification = Notification.Name("ReachabilityChangedNotification")

func callback(_: SCNetworkReachability, flags _: SCNetworkReachabilityFlags, info: UnsafeMutableRawPointer?) {
    guard let info = info else { return }
    let reachability = Unmanaged<Reachability>.fromOpaque(info).takeUnretainedValue()

    DispatchQueue.main.async {
        reachability.reachabilityChanged()
    }
}

open class Reachability {
    public typealias NetworkReachable = (Reachability) -> Void
    public typealias NetworkUnreachable = (Reachability) -> Void

    public enum NetworkStatus: CustomStringConvertible {
        case notReachable, reachableViaWiFi, reachableViaWWAN

        public var description: String {
            switch self {
            case .reachableViaWWAN: return "Cellular"
            case .reachableViaWiFi: return "WiFi"
            case .notReachable: return "No Connection"
            }
        }
    }

    open var whenReachable: NetworkReachable?
    open var whenUnreachable: NetworkUnreachable?
    open var reachableOnWWAN: Bool
    // The notification center on which "reachability changed" events are being posted
    open var notificationCenter: NotificationCenter = .default

    open var currentReachabilityString: String {
        return "\(currentReachabilityStatus)"
    }

    open var currentReachabilityStatus: NetworkStatus {
        guard isReachable else { return .notReachable }
        if isReachableViaWiFi {
            return .reachableViaWiFi
        }
        if isRunningOnDevice {
            return .reachableViaWWAN
        }
        return .notReachable
    }

    fileprivate var previousFlags: SCNetworkReachabilityFlags?
    fileprivate var isRunningOnDevice: Bool = {
        #if targetEnvironment(simulator)
            return false
        #else
            return true
        #endif
    }()

    fileprivate var notifierRunning = false
    fileprivate var reachabilityRef: SCNetworkReachability?
    fileprivate let reachabilitySerialQueue = DispatchQueue(label: "uk.co.ashleymills.reachability")
    public required init(reachabilityRef: SCNetworkReachability) {
        reachableOnWWAN = true
        self.reachabilityRef = reachabilityRef
//        do {
//            try self.startNotifier()
//        }
//        catch{
//            debugPrint("error in starting notifier for internet connection check callback")
//        }
    }

    public convenience init?(hostname: String) {
        guard let ref = SCNetworkReachabilityCreateWithName(nil, hostname) else { return nil }
        self.init(reachabilityRef: ref)
//        do {
//            try self.startNotifier()
//        }
//        catch{
//            debugPrint("error in starting notifier for internet connection check callback")
//        }
    }

    public convenience init?() {
        var zeroAddress = sockaddr()
        zeroAddress.sa_len = UInt8(MemoryLayout<sockaddr>.size)
        zeroAddress.sa_family = sa_family_t(AF_INET)
        guard let ref: SCNetworkReachability = withUnsafePointer(to: &zeroAddress, {
            SCNetworkReachabilityCreateWithAddress(nil, UnsafePointer($0))
        }) else { return nil }
        self.init(reachabilityRef: ref)
//        do {
//            try self.startNotifier()
//        }
//        catch{
//            debugPrint("error in starting notifier for internet connection check callback")
//        }
    }

    deinit {
        stopNotifier()

        reachabilityRef = nil
        whenReachable = nil
        whenUnreachable = nil
    }
}

public extension Reachability {
    // MARK: - *** Notifier methods ***

    func startNotifier() throws {
        guard let reachabilityRef = reachabilityRef, !notifierRunning else { return }
        var context = SCNetworkReachabilityContext(version: 0, info: nil, retain: nil, release: nil, copyDescription: nil)
        context.info = UnsafeMutableRawPointer(Unmanaged<Reachability>.passUnretained(self).toOpaque())
        if !SCNetworkReachabilitySetCallback(reachabilityRef, callback, &context) {
            stopNotifier()
            throw ReachabilityError.unableToSetCallback
        }
        if !SCNetworkReachabilitySetDispatchQueue(reachabilityRef, reachabilitySerialQueue) {
            stopNotifier()
            throw ReachabilityError.unableToSetDispatchQueue
        }
        // Perform an initial check
        reachabilitySerialQueue.async {
            self.reachabilityChanged()
        }
        notifierRunning = true
    }

    func stopNotifier() {
        defer { notifierRunning = false }
        guard let reachabilityRef = reachabilityRef else { return }
        SCNetworkReachabilitySetCallback(reachabilityRef, nil, nil)
        SCNetworkReachabilitySetDispatchQueue(reachabilityRef, nil)
    }

    // MARK: - *** Connection test methods ***

    var isReachable: Bool {
        guard isReachableFlagSet else { return false }
        if isConnectionRequiredAndTransientFlagSet {
            return false
        }
        if isRunningOnDevice {
            if isOnWWANFlagSet && !reachableOnWWAN {
                // We don't want to connect when on 3G.
                return false
            }
        }
        return true
    }

    var isReachableViaWWAN: Bool {
        // Check we're not on the simulator, we're REACHABLE and check we're on WWAN
        return isRunningOnDevice && isReachableFlagSet && isOnWWANFlagSet
    }

    var isReachableViaWiFi: Bool {
        // Check we're reachable
        guard isReachableFlagSet else { return false }
        // If reachable we're reachable, but not on an iOS device (i.e. simulator), we must be on WiFi
        guard isRunningOnDevice else { return true }
        // Check we're NOT on WWAN
        return !isOnWWANFlagSet
    }

    var description: String {
        let runningOnDevice = isRunningOnDevice ? (isOnWWANFlagSet ? "W" : "-") : "X"
        let reachableFlagSet = isReachableFlagSet ? "R" : "-"
        let connectionRequiredFlagSet = isConnectionRequiredFlagSet ? "c" : "-"
        let transientConnectionFlagSet = isTransientConnectionFlagSet ? "t" : "-"
        let interventionRequiredFlagSet = isInterventionRequiredFlagSet ? "i" : "-"
        let connectionOnTrafficFlagSet = isConnectionOnTrafficFlagSet ? "C" : "-"
        let connectionOnDemandFlagSet = isConnectionOnDemandFlagSet ? "D" : "-"
        let localAddressFlagSet = isLocalAddressFlagSet ? "l" : "-"
        let directFlagSet = isDirectFlagSet ? "d" : "-"
        return "\(runningOnDevice)\(reachableFlagSet) \(connectionRequiredFlagSet)\(transientConnectionFlagSet)\(interventionRequiredFlagSet)\(connectionOnTrafficFlagSet)\(connectionOnDemandFlagSet)\(localAddressFlagSet)\(directFlagSet)"
    }
}

private extension Reachability {
    func reachabilityChanged() {
        let flags = reachabilityFlags
        guard previousFlags != flags else { return }
        let block = isReachable ? whenReachable : whenUnreachable
        block?(self)
        notificationCenter.post(name: reachabilityChangedNotification, object: self)
        previousFlags = flags
    }

    var isOnWWANFlagSet: Bool {
        #if os(iOS)
            return reachabilityFlags.contains(.isWWAN)
        #else
            return false
        #endif
    }

    var isReachableFlagSet: Bool {
        return reachabilityFlags.contains(.reachable)
    }

    var isConnectionRequiredFlagSet: Bool {
        return reachabilityFlags.contains(.connectionRequired)
    }

    var isInterventionRequiredFlagSet: Bool {
        return reachabilityFlags.contains(.interventionRequired)
    }

    var isConnectionOnTrafficFlagSet: Bool {
        return reachabilityFlags.contains(.connectionOnTraffic)
    }

    var isConnectionOnDemandFlagSet: Bool {
        return reachabilityFlags.contains(.connectionOnDemand)
    }

    var isConnectionOnTrafficOrDemandFlagSet: Bool {
        return !reachabilityFlags.intersection([.connectionOnTraffic, .connectionOnDemand]).isEmpty
    }

    var isTransientConnectionFlagSet: Bool {
        return reachabilityFlags.contains(.transientConnection)
    }

    var isLocalAddressFlagSet: Bool {
        return reachabilityFlags.contains(.isLocalAddress)
    }

    var isDirectFlagSet: Bool {
        return reachabilityFlags.contains(.isDirect)
    }

    var isConnectionRequiredAndTransientFlagSet: Bool {
        return reachabilityFlags.intersection([.connectionRequired, .transientConnection]) == [.connectionRequired, .transientConnection]
    }

    var reachabilityFlags: SCNetworkReachabilityFlags {
        guard let reachabilityRef = reachabilityRef else { return SCNetworkReachabilityFlags() }
        var flags = SCNetworkReachabilityFlags()
        let gotFlags = withUnsafeMutablePointer(to: &flags) {
            SCNetworkReachabilityGetFlags(reachabilityRef, UnsafeMutablePointer($0))
        }
        if gotFlags {
            return flags
        } else {
            return SCNetworkReachabilityFlags()
        }
    }
}
