//
//  ImagePicker.swift
//  BeInBeOutAMCO
//
//  Created by Chintan on 21/05/19.
//

import AVFoundation
import Foundation
import Photos
import UIKit

public protocol ImagePickerDelegate: AnyObject {
    func imagePicked(image: UIImage?)
}

open class ImagePicker: NSObject, UIImagePickerControllerDelegate, UINavigationControllerDelegate {
    private let pickerController: UIImagePickerController
    private weak var viewController: UIViewController?
    private weak var imagePickerDelegate: ImagePickerDelegate?
    private var allowEditing: Bool = false

    // MARK: -

    public init(viewController: UIViewController, imagePickerDelegate: ImagePickerDelegate, allowEditing: Bool = false) {
        // Pass allowEditing = true if you want to allow user to edit picked image using iOS's default editing functionality
        pickerController = UIImagePickerController()

        super.init()

        self.viewController = viewController
        self.imagePickerDelegate = imagePickerDelegate
        self.allowEditing = allowEditing

        pickerController.delegate = self
        pickerController.allowsEditing = allowEditing
        pickerController.mediaTypes = ["public.image"]
    }

    // MARK: - Methods

    /// Presents aleart dialog with all options.
    public func present() {
        DispatchQueue.main.async {
            self.present([.camera, .savedPhotosAlbum, .photoLibrary])
        }
    }

    /// Presents aleart dialog with specified options.
    ///
    /// - Parameter arrSourceTypes: Array of required SourceTypes.
    public func present(_ arrSourceTypes: [UIImagePickerController.SourceType]) {
        if arrSourceTypes.isEmpty {
            debugPrint("Error: Source type array must contain at least one element.")

            return
        }

        let alertController = UIAlertController(title: nil, message: nil, preferredStyle: .actionSheet)

        var title = ""

        for sourceType in arrSourceTypes {
            switch sourceType {
            // Take Photo
            case .camera:
                title = NSLocalizedString("take_photo", comment: "")

            // Camera Roll
            case .savedPhotosAlbum:
                title = NSLocalizedString("img_from_camera_roll_msg", comment: "")

            // Photo Library
            case .photoLibrary:
                title = NSLocalizedString("img_from_library_msg", comment: "")

            default:
                debugPrint("Error: Invalid source type.")
            }

            if let action = getAction(for: sourceType, title: title) {
                alertController.addAction(action)
            }
        }

        if alertController.actions.isEmpty {
            debugPrint("Error: This device doesn't support passed source type(s).")

            return
        }

        alertController.addAction(UIAlertAction(title: NSLocalizedString("cancel", comment: ""), style: .cancel, handler: nil))
        DispatchQueue.main.async {
            self.viewController?.present(alertController, animated: true)
        }
    }

    private func getAction(for type: UIImagePickerController.SourceType, title: String) -> UIAlertAction? {
        guard UIImagePickerController.isSourceTypeAvailable(type) else {
            return nil
        }

        return UIAlertAction(title: title, style: .default) { _ in
            self.pickerController.sourceType = type

            switch type {
            case .camera:
                if AVCaptureDevice.authorizationStatus(for: AVMediaType.video) == AVAuthorizationStatus.authorized {
                    // Already Authorized
                    DispatchQueue.main.async {
                        self.viewController?.present(self.pickerController, animated: true)
                    }
                } else {
                    AVCaptureDevice.requestAccess(for: AVMediaType.video, completionHandler: { (granted: Bool) in
                        if granted {
                            // User granted the permission
                            DispatchQueue.main.async {
                                self.viewController?.present(self.pickerController, animated: true)
                            }
                        } else {
                            // User rejected the permission. Asking user to enable permission from Settings.
                            self.showEnablePermissionAlert(title: NSLocalizedString("camera_permission", comment: ""),
                                                           message: NSLocalizedString("enable_camera_permission_msg", comment: ""))
                        }
                    })
                }

            case .savedPhotosAlbum, .photoLibrary:
                if PHPhotoLibrary.authorizationStatus() == .authorized {
                    DispatchQueue.main.async {
                        // Already Authorized
                        self.viewController?.present(self.pickerController, animated: true)
                    }
                } else {
                    PHPhotoLibrary.requestAuthorization { status in
                        if status == .authorized {
                            DispatchQueue.main.async {
                                // Already Authorized
                                self.viewController?.present(self.pickerController, animated: true)
                            }
                        } else {
                            DispatchQueue.main.async {
                                // User rejected the permission. Asking user to enable permission from Settings.
                                self.showEnablePermissionAlert(title: NSLocalizedString("photos_permission", comment: ""),
                                                               message: NSLocalizedString("enable_photos_permission_msg", comment: ""))
                            }
                        }
                    }
                }

            default:
                debugPrint("Error: Invalid source type clicked.")
            }
        }
    }

    func showEnablePermissionAlert(title: String, message: String) {
        DispatchQueue.main.async {
            let alertController = UIAlertController(title: title, message: message, preferredStyle: .alert)

            alertController.addAction(UIAlertAction(title: NSLocalizedString("no_thanks", comment: ""), style: .cancel) { _ in
                self.imagePickerDelegate?.imagePicked(image: nil)
            })

            alertController.addAction(UIAlertAction(title: NSLocalizedString("open_settings", comment: ""), style: .default) { _ in
                if let appSettings = URL(string: UIApplication.openSettingsURLString) {
                    UIApplication.shared.open(appSettings, options: [:], completionHandler: nil)
                }
            })
            DispatchQueue.main.async {
                self.viewController?.present(alertController, animated: true, completion: nil)
            }
        }
    }

    // MARK: - Image Picker Controller Delegate Methods

    public func imagePickerController(_ picker: UIImagePickerController, didFinishPickingMediaWithInfo info: [UIImagePickerController.InfoKey: Any]) {
        DispatchQueue.main.async {
            guard let image = info[self.allowEditing ? .editedImage : .originalImage] as? UIImage else {
                self.imagePickerDelegate?.imagePicked(image: nil)

                return
            }

            self.imagePickerDelegate?.imagePicked(image: image)
            picker.dismiss(animated: true, completion: nil)
        }
    }

    public func imagePickerControllerDidCancel(_ picker: UIImagePickerController) {
        DispatchQueue.main.async {
            picker.dismiss(animated: true, completion: nil)

            self.imagePickerDelegate?.imagePicked(image: nil)
        }
    }
}
