//
//  APIManager.swift
//  MVVM Sample iOS
//

import Alamofire
import Foundation

struct ValidationError {
    var field: String
    var message: String
}

class APIManager {
    // MARK: HEADER

    static let headers: HTTPHeaders = [
//        "ApiKey": Constant.apiKey,
//        "AppVersion": Helper.getVersion(),
//        "AppInternalVersion": Helper.getBuildVersion(),
//        "DeviceType": "ios",
        "Accept": "application/json; charset=utf-8",
        "Content-Type": "application/json; charset=utf-8",
        "accept-language": "en"
    ]

    // MARK: API CALL FUNCTIONS

    /**
      Calls a web service API using Alamofire.

      - Parameters:
          - apiUrl: The URL of the API endpoint you want to call.
          - params: A dictionary containing additional parameters for the API request.
          - encodableParams: An optional Encodable object to be sent as the request body.
          - resultType: The expected response type from the API.
          - customHeader: Custom HTTP headers for the API request.
          - httpMethod: The HTTP method to be used for the API request.
          - encoding: The encoding method for the parameters in the API request.
          - completion: A closure called when the API request is completed.
                         It takes a Result object containing either the response data or an error.
     */
    static func callWebService<E: Encodable, D: Decodable>(
        apiUrl: String,
        params: [String: Any],
        encodableParams: E? = nil,
        resultType: D.Type,
        customHeader: HTTPHeaders?,
        httpMethod: HTTPMethod,
        encoding: ParameterEncoding = JSONEncoding.default,
        completion: @escaping (Result<D, BaseError>) -> Void
    ) {
        if Helper.checkInternetConnection(giveErrorMessage: true) {
            AF.session.configuration.timeoutIntervalForRequest = 300
            AF.request(apiUrl, method: httpMethod, parameters: params, encoding: encoding, headers: headers).responseDecodable(of: resultType) { response in
                debugPrint(apiUrl, params, response)
                switch response.result {
                case let .success(parsedResponse):

                    if response.response?.statusCode == HTTPStatusCode.okay.rawValue {
                        completion(.success(parsedResponse))
                    } else {
                        let apiError = handleAPIError(response: response)
                        completion(.failure(apiError))
                    }

                case .failure:
                    let apiError = handleAPIError(response: response)
                    completion(.failure(apiError))
                }
            }
        } else {
            completion(.failure(.noInternetConnection))
        }
    }

    static func callWebServiceWithMultipartData<E: Encodable, D: Decodable>(
        apiUrl: String,
        params: [String: Any]?,
        encodableParams: E? = nil,
        multipartData: [MultiPartData],
        resultType: D.Type,
        customHeader: HTTPHeaders?,
        httpMethod: HTTPMethod,
        completion: @escaping (Result<D, BaseError>) -> Void
    ) {
        if Helper.checkInternetConnection(giveErrorMessage: true) {
            AF.session.configuration.timeoutIntervalForRequest = 300

            var headers = self.headers

            // Below token should be sent only to those APIs where it is required.
            if !AccountRepository.getAccessToken().isEmpty {
                headers["Authorization"] = "Bearer \(AccountRepository.getAccessToken())"
            }

            if let additionalHeaders = customHeader {
                headers = HTTPHeaders(headers.dictionary.merging(additionalHeaders.dictionary) { _, new in new })
            }

            let multipartFormData = MultipartFormData()

            for multipart in multipartData {
                if let data = multipart.data {
                    multipartFormData.append(data, withName: multipart.keyName, fileName: multipart.fileName, mimeType: multipart.mimeType)
                }
            }

            if let encodableParams = encodableParams {
                do {
                    let encodedParams = try JSONEncoder().encode(encodableParams)
                    if let encodedDictionary = try JSONSerialization.jsonObject(with: encodedParams, options: []) as? [String: Any] {
                        for (key, value) in encodedDictionary {
                            if let data = "\(value)".data(using: .utf8) {
                                multipartFormData.append(data, withName: key)
                            }
                        }
                    }
                } catch {
                    completion(.failure(.parsingError(statusCode: 0, message: "something_went_wrong".localized)))
                    return
                }
            }

            if let params = params {
                for (key, value) in params {
                    if let data = "\(value)".data(using: .utf8) {
                        multipartFormData.append(data, withName: key)
                    }
                }
            }

            AF.upload(multipartFormData: multipartFormData, to: apiUrl, method: httpMethod, headers: headers).responseDecodable(of: resultType) { response in
                debugPrint(apiUrl, params ?? [:], response)
                switch response.result {
                case let .success(parsedResponse):

                    if response.response?.statusCode == HTTPStatusCode.okay.rawValue {
                        completion(.success(parsedResponse))
                    } else {
                        let apiError = handleAPIError(response: response)
                        completion(.failure(apiError))
                    }

                case .failure:
                    let apiError = handleAPIError(response: response)
                    completion(.failure(apiError))
                }
            }
        } else {
            completion(.failure(.noInternetConnection))
        }
    }
}

extension APIManager {
    static func callWebServiceWithManualParsing<E: Encodable, D: Decodable>(
        apiUrl: String,
        params: [String: Any],
        encodableParams: E? = nil,
        resultType _: D.Type,
        customHeader: HTTPHeaders?,
        httpMethod: HTTPMethod,
        encoding: ParameterEncoding = JSONEncoding.default,
        completion: @escaping (Result<[String: Any], BaseError>) -> Void
    ) {
        if Helper.checkInternetConnection(giveErrorMessage: true) {
            AF.session.configuration.timeoutIntervalForRequest = 300

            var headers = self.headers

            // Below token should be sent only to those APIs where it is required.
            if !AccountRepository.getAccessToken().isEmpty {
                headers["Authorization"] = "Bearer \(AccountRepository.getAccessToken())"
            }

            if let additionalHeaders = customHeader {
                headers = HTTPHeaders(headers.dictionary.merging(additionalHeaders.dictionary) { _, new in new })
            }

            var parameters = params

            if let encodableParams = encodableParams {
                do {
                    let encodedParams = try JSONEncoder().encode(encodableParams)
                    if let encodedDictionary = try JSONSerialization.jsonObject(with: encodedParams, options: []) as? [String: Any] {
                        parameters.merge(encodedDictionary, uniquingKeysWith: { _, new in new })
                    }
                } catch {
                    completion(.failure(.parsingError(statusCode: 0, message: "something_went_wrong".localized)))
                    return
                }
            }

            AF.request(apiUrl, method: httpMethod, parameters: parameters, encoding: encoding, headers: headers).responseData { response in

                debugPrint(apiUrl, parameters, response)
                switch response.result {
                case let .success(value):
                    do {
                        let json = try JSONSerialization.jsonObject(with: value, options: [])
                        if let dictionary = json as? [String: Any] {
                            if response.response?.statusCode == HTTPStatusCode.okay.rawValue {
                                completion(.success(dictionary))
                            } else {
                                completion(.failure(handleAPIError(response: response)))
                            }
                        } else {
                            completion(.failure(.parsingError(statusCode: 0, message: "Failed to parse API response.")))
                        }
                    } catch {
                        completion(.failure(.parsingError(statusCode: 0, message: "Failed to parse API response: \(error)")))
                    }
                case .failure:
                    completion(.failure(handleAPIError(response: response)))
                }
            }
        } else {
            completion(.failure(.noInternetConnection))
        }
    }

    static func callWebServiceWithMultipartDataWithManualParsing(
        apiUrl: String,
        params: [String: Any]?,
        multipartData: [MultiPartData],
        customHeader: HTTPHeaders?,
        httpMethod: HTTPMethod,
        completion: @escaping (Result<[String: Any], BaseError>) -> Void
    ) {
        if Helper.checkInternetConnection(giveErrorMessage: true) {
            AF.session.configuration.timeoutIntervalForRequest = 300

            var headers = self.headers

            // Below token should be sent only to those APIs where it is required.
            if !AccountRepository.getAccessToken().isEmpty {
                headers["Authorization"] = "Bearer \(AccountRepository.getAccessToken())"
            }

            if let additionalHeaders = customHeader {
                headers = HTTPHeaders(headers.dictionary.merging(additionalHeaders.dictionary) { _, new in new })
            }

            let parameters = params ?? [:]

            let multipartFormData = MultipartFormData()

            for multipart in multipartData {
                if let data = multipart.data {
                    multipartFormData.append(data, withName: multipart.keyName, fileName: multipart.fileName, mimeType: multipart.mimeType)
                }
            }

            AF.upload(multipartFormData: multipartFormData, to: apiUrl, method: httpMethod, headers: headers).responseData { response in
                debugPrint(apiUrl, parameters, response)
                switch response.result {
                case let .success(value):
                    do {
                        let json = try JSONSerialization.jsonObject(with: value, options: [])
                        if let dictionary = json as? [String: Any] {
                            if response.response?.statusCode == HTTPStatusCode.okay.rawValue {
                                completion(.success(dictionary))
                            } else {
                                completion(.failure(handleAPIError(response: response)))
                            }
                        } else {
                            completion(.failure(.parsingError(statusCode: 0, message: "Failed to parse API response.")))
                        }
                    } catch {
                        completion(.failure(.parsingError(statusCode: 0, message: "Failed to parse API response: \(error)")))
                    }
                case .failure:
                    completion(.failure(handleAPIError(response: response)))
                }
            }
        } else {
            completion(.failure(.noInternetConnection))
        }
    }
}

// MARK: PRIVATE HELPER FUNCTIONS

extension APIManager {
    private static func handleAPIError<D: Decodable>(response: DataResponse<D, AFError>) -> BaseError {
        let statusCode = response.response?.statusCode ?? 0
        var errorMessage = "something_went_wrong".localized

        var validationError: [ValidationError] = []

        if let data = response.data, let errorDict = try? JSONSerialization.jsonObject(with: data, options: []) as? [String: Any] {
            if let message = errorDict["message"] as? String {
                errorMessage = message
            }

            if let errors = errorDict["errors"] as? [[String: Any]] {
                for err in errors {
                    let field = err["field"] as? String ?? ""
                    let message = err["message"] as? String ?? ""
                    validationError.append(ValidationError(field: field, message: message))
                }
            }
        }

        var apiError: BaseError

        switch statusCode {
        case HTTPStatusCode.inputValidationSecondError.rawValue:
            apiError = .inputValidationError(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.inputValidationFirstError.rawValue:
            apiError = .inputValidationError(statusCode: statusCode, message: errorMessage, validationError: validationError)
        case HTTPStatusCode.parsingError.rawValue:
            apiError = .parsingError(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.unauthorized.rawValue:
            apiError = .unauthorized(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.forcefullyLogout.rawValue:
            apiError = .forcefullyLogout(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.resourceNotFound.rawValue:
            apiError = .resourceNotFound(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.internalServerError.rawValue:
            apiError = .internalServerError(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.badGateway.rawValue:
            apiError = .badGateway(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.serviceUnavailable.rawValue:
            apiError = .serviceUnavailable(statusCode: statusCode, message: errorMessage)
        default:
            apiError = .unknown
        }

        return apiError
    }

    private static func handleAPIError(response: AFDataResponse<Any>) -> BaseError {
        let statusCode = response.response?.statusCode ?? 0
        var errorMessage = "something_went_wrong".localized

        var validationError: [ValidationError] = []

        if let data = response.data, let errorDict = try? JSONSerialization.jsonObject(with: data, options: []) as? [String: Any] {
            if let message = errorDict["message"] as? String {
                errorMessage = message
            }

            if let errors = errorDict["errors"] as? [[String: Any]] {
                for err in errors {
                    let field = err["field"] as? String ?? ""
                    let message = err["message"] as? String ?? ""
                    validationError.append(ValidationError(field: field, message: message))
                }
            }
        }

        var apiError: BaseError

        switch statusCode {
        case HTTPStatusCode.inputValidationSecondError.rawValue:
            apiError = .inputValidationError(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.inputValidationFirstError.rawValue:
            apiError = .inputValidationError(statusCode: statusCode, message: errorMessage, validationError: validationError)
        case HTTPStatusCode.parsingError.rawValue:
            apiError = .parsingError(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.unauthorized.rawValue:
            apiError = .unauthorized(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.forcefullyLogout.rawValue:
            apiError = .forcefullyLogout(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.resourceNotFound.rawValue:
            apiError = .resourceNotFound(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.internalServerError.rawValue:
            apiError = .internalServerError(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.badGateway.rawValue:
            apiError = .badGateway(statusCode: statusCode, message: errorMessage)
        case HTTPStatusCode.serviceUnavailable.rawValue:
            apiError = .serviceUnavailable(statusCode: statusCode, message: errorMessage)
        default:
            apiError = .unknown
        }

        return apiError
    }
}

extension APIManager {
    // MARK: NORMAL API CALL

    // MARK: 1.ENCODED REQUEST WITH MENUAL PARSING

    static func callWebServiceWithEncodedRequestManualParsing<E: Encodable>(
        apiUrl: String,
        params: [String: Any],
        encodableParams: E,
        customHeader: HTTPHeaders?,
        httpMethod: HTTPMethod,
        encoding: ParameterEncoding = JSONEncoding.default,
        completion: @escaping (Result<[String: Any], BaseError>) -> Void
    ) {
        var parameters = params

        do {
            let encodedParams = try JSONEncoder().encode(encodableParams)
            if let encodedDictionary = try JSONSerialization.jsonObject(with: encodedParams, options: []) as? [String: Any] {
                parameters.merge(encodedDictionary, uniquingKeysWith: { _, new in new })
            }
        } catch {
            completion(.failure(.parsingError(statusCode: 0, message: "something_went_wrong".localized)))
            return
        }

        callWebServiceWithRawParamRequestManualParsing(apiUrl: apiUrl, params: parameters, customHeader: customHeader, httpMethod: httpMethod, encoding: encoding) { response in

            switch response {
            case let .success(success):
                completion(.success(success))
            case let .failure(failure):
                completion(.failure(failure))
            }
        }
    }

    // MARK: 2.ENCODED REQUEST WITH JSON PARSING

    static func callWebServiceEncodedRequestJsonParsing<E: Encodable, D: Decodable>(
        apiUrl: String,
        params: [String: Any],
        encodableParams: E,
        resultType: D.Type,
        customHeader: HTTPHeaders?,
        httpMethod: HTTPMethod,
        encoding: ParameterEncoding = URLEncoding.default,
        completion: @escaping (Result<D, BaseError>) -> Void
    ) {
        var parameters = params

        do {
            let encodedParams = try JSONEncoder().encode(encodableParams)
            if let encodedDictionary = try JSONSerialization.jsonObject(with: encodedParams, options: []) as? [String: Any] {
                parameters.merge(encodedDictionary, uniquingKeysWith: { _, new in new })
            }
        } catch {
            completion(.failure(.parsingError(statusCode: 0, message: "something_went_wrong".localized)))
            return
        }

        callWebServiceWithRawParamRequestJsonParsing(apiUrl: apiUrl, params: parameters, resultType: resultType, customHeader: customHeader, httpMethod: httpMethod, encoding: encoding) { response in
            switch response {
            case let .success(success):
                completion(.success(success))
            case let .failure(failure):
                completion(.failure(failure))
            }
        }
    }

    // MARK: 3.RAW PARAM REQUEST WITH MENUAL PARSING

    static func callWebServiceWithRawParamRequestManualParsing(
        apiUrl: String,
        params: [String: Any],
        customHeader: HTTPHeaders?,
        httpMethod: HTTPMethod,
        encoding: ParameterEncoding = URLEncoding.default,
        completion: @escaping (Result<[String: Any], BaseError>) -> Void
    ) {
        if Helper.checkInternetConnection(giveErrorMessage: true) {
            AF.session.configuration.timeoutIntervalForRequest = 300

            var headers = self.headers

            // Below token should be sent only to those APIs where it is required.
            if !AccountRepository.getAccessToken().isEmpty {
                headers["Authorization"] = "Bearer \(AccountRepository.getAccessToken())"
            }

            if let additionalHeaders = customHeader {
                headers = HTTPHeaders(headers.dictionary.merging(additionalHeaders.dictionary) { _, new in new })
            }

            AF.request(apiUrl, method: httpMethod, parameters: params, encoding: encoding, headers: headers).responseData { response in

                debugPrint(apiUrl, params, response)
                switch response.result {
                case let .success(value):
                    do {
                        let json = try JSONSerialization.jsonObject(with: value, options: [])
                        if let dictionary = json as? [String: Any] {
                            if response.response?.statusCode == HTTPStatusCode.okay.rawValue {
                                completion(.success(dictionary))
                            } else {
                                completion(.failure(handleAPIError(response: response)))
                            }
                        } else {
                            completion(.failure(.parsingError(statusCode: 0, message: "Failed to parse API response.")))
                        }
                    } catch {
                        completion(.failure(.parsingError(statusCode: 0, message: "Failed to parse API response: \(error)")))
                    }
                case .failure:
                    completion(.failure(handleAPIError(response: response)))
                }
            }
        } else {
            completion(.failure(.noInternetConnection))
        }
    }

    // MARK: 4.RAW PARAM REQUEST WITH JSON PARSING

    static func callWebServiceWithRawParamRequestJsonParsing<D: Decodable>(
        apiUrl: String,
        params: [String: Any],
        resultType: D.Type,
        customHeader: HTTPHeaders?,
        httpMethod: HTTPMethod,
        encoding: ParameterEncoding = JSONEncoding.default,
        completion: @escaping (Result<D, BaseError>) -> Void
    ) {
        if Helper.checkInternetConnection(giveErrorMessage: true) {
            AF.session.configuration.timeoutIntervalForRequest = 300

            var headers = self.headers

            // Below token should be sent only to those APIs where it is required.
//            if !AccountRepository.getAccessToken().isEmpty {
//                headers["Authorization"] = "Bearer \(AccountRepository.getAccessToken())"
//            }

            if let additionalHeaders = customHeader {
                headers = HTTPHeaders(headers.dictionary.merging(additionalHeaders.dictionary) { _, new in new })
            }

            AF.request(apiUrl, method: httpMethod, parameters: params, encoding: encoding, headers: headers).responseDecodable(of: resultType) { response in
                debugPrint(apiUrl, params, response)
                switch response.result {
                case let .success(parsedResponse):

                    if response.response?.statusCode == HTTPStatusCode.okay.rawValue {
                        completion(.success(parsedResponse))
                    } else {
                        let apiError = handleAPIError(response: response)
                        completion(.failure(apiError))
                    }

                case .failure:
                    let apiError = handleAPIError(response: response)
                    completion(.failure(apiError))
                }
            }
        } else {
            completion(.failure(.noInternetConnection))
        }
    }
}
