package com.amco.bein_beout.model

import android.app.Application
import android.util.Log
import androidx.lifecycle.AndroidViewModel
import com.amco.bein_beout.data.entity.APIResult


import com.amco.bein_beout.data.entity.User
import com.amco.bein_beout.data.repository.AccountRepository
import com.amco.bein_beout.local.PreferenceManager
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import kotlin.coroutines.CoroutineContext


class AccountViewModel(application: Application) : AndroidViewModel(application) {
    private val parentJob = Job()
    private val coroutineContext: CoroutineContext get() = parentJob + Dispatchers.Main
    private val scope: CoroutineScope = CoroutineScope(coroutineContext)
    private val accountRepository = AccountRepository(application)
    private val preferenceManager = PreferenceManager(application)
    var user = User()

    /**
     * this method is used for login user.
     */
    fun loginApi(email: String, password: String, listener: (APIResult<String>) -> Unit) {
        scope.launch {
            accountRepository.loginApi(email,password) {
                when (it) {
                    is APIResult.Success-> {
                        Log.e("message", it.message.toString())
                         user = it.data
                        preferenceManager.accessToken = user.token!!
                        preferenceManager.userName =user.userFirstName!!+" " + user.userLastName
                        listener(APIResult.Success("", it.message))
                    }

                    is APIResult.Failure -> {
                        listener(it)
                    }

                    APIResult.InProgress -> {
                        listener(APIResult.InProgress)
                    }
                }
            }
        }
    }

    /**
     * this method is used for create an account.
     */
    fun signUpApi(user1: User, listener: (APIResult<String>) -> Unit) {
        scope.launch {
            accountRepository.signUpApi(user1) {
                when (it) {
                    is APIResult.Success-> {
                        Log.e("message", it.message.toString())
                         user = it.data
                        preferenceManager.accessToken = user.token!!
                        preferenceManager.userName =  if (user.username!!.isNotEmpty()) user.username!! else user.userFirstName!!+" " + user.userLastName
                        listener(APIResult.Success("", it.message))
                    }
                    is APIResult.Failure -> {
                        listener(it)
                    }
                    APIResult.InProgress -> {
                        listener(APIResult.InProgress)
                    }
                }
            }
        }
    }

    /**
     * this method is used for reset password.
     */
    fun resetPasswordApi(
        email: String,
        password: String,
        token: String,
        listener: (APIResult<String>) -> Unit
    ) {
        scope.launch {
            accountRepository.resetPasswordApi(email, password, token) {
                when (it) {
                    is APIResult.Success -> {
                        Log.e("message", it.message.toString())
                        listener(APIResult.Success("", it.message))
                    }

                    is APIResult.Failure -> {
                        listener(it)
                    }

                    APIResult.InProgress -> {
                        listener(APIResult.InProgress)
                    }
                }
            }
        }
    }

    /**
     * this method is used forgot password.
     */
    fun forgotPasswordApi(email: String, listener: (APIResult<String>) -> Unit) {
        scope.launch {
            accountRepository.forgotPasswordApi(email) {
                when (it) {
                    is APIResult.Success<*> -> {
                        Log.e("message", it.message.toString())
                        listener(APIResult.Success("", it.message))
                    }

                    is APIResult.Failure -> {
                        listener(it)
                    }

                    APIResult.InProgress -> {
                        listener(APIResult.InProgress)
                    }
                }
            }
        }
    }

    /**
     * Set And Get Token
     */
    fun getAccessToken() = preferenceManager.accessToken
    fun logOut() = preferenceManager.clearSession()
    fun getUserName() = preferenceManager.userName
}